package winterwell.j4square;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.json.JSONException;
import org.json.JSONObject;

import winterwell.jtwitter.Twitter.IHttpClient;
import winterwell.jtwitter.URLConnectionHttpClient;

/**
 * Java wrapper for the foursquare API version {@value #version}
 * <p>
 *  OAuth notes: http://groups.google.com/group/foursquare-api/web/oauth
 *  API notes: http://groups.google.com/group/foursquare-api/web/api-documentation
 *
 * <h4>Copyright and License</h4>
 * This code is copyright (c) Winterwell Associates 2010.
 * It is released as
 * open-source under the LGPL license with Attribution (you must give
 * Winterwell credit when using it via a link to our website).
 * This code
 * comes with no warranty or support.
 *
 * @author Daniel Winterstein
 *
 */
public class J4Square {

	/*
	 * Request Token URL: http://foursquare.com/oauth/request_token
Access Token URL: http://foursquare.com/oauth/access_token
Authorize URL: http://foursquare.com/oauth/authorize

We currently support hmac-sha1 signed requests
Application Name: sodash
Callback URL: http://soda.sh/oauth
Key: 4PMR03RC2CIFTT1OM3CUK4ZHR05TQHGC3FRZUYEZU4EXX0LY
Secret: RSEUN315WKNC4OKX0ZK2YQUFUN4REM30RFIKPIZQPS5I2HNT

	 * 
	 * 
	 */
	public static final String version = "0.1";
	
	private final IHttpClient http;
	
	private String API_URL = "http://api.foursquare.com/v1/";

	private Boolean sendToTwitter;

	private Boolean sendToFacebook;

	private boolean hide;
	
	public List<CheckIn> getHistory() {
		Map<String,Object> vars = getStandardishParameters();
		return getCheckIns(API_URL+"history.json", vars);
	}
	
	/**
	 * @return Recent check-ins from friends
	 */
	public List<CheckIn> getCheckIns() {
		Map<String,Object> vars = getStandardishParameters();
		return getCheckIns(API_URL+"checkins.json", vars);
	}
	
	/*
	 * 	vid - (optional, not necessary if you are 'shouting' or have a venue name). ID of the venue where you want to check-in
	venue - (optional, not necessary if you are 'shouting' or have a vid) 
	if you don't have a venue ID or would rather prefer a 'venueless' checkin, 
	pass the venue name as a string using this parameter. 
	it will become an 'orphan' (no address or venueid but with geolat, geolong)
	shout - (optional) a message about your check-in. the maximum length of this field is 140 characters
	private - (optional). "1" means "don't show your friends". "0" means "show everyone"
	twitter - (optional, defaults to the user's setting). "1" means "send to Twitter". "0" means "don't send to Twitter"
	facebook - (optional, defaults to the user's setting). "1" means "send to Facebook". "0" means "don't send to Facebook"
	geolat - (optional, but recommended)
	geolong - (optional, but recommended)

	 */
	public CheckIn checkIn(Venue venue, String shout) {
		Map vars = getStandardishParameters();
		if (venue!=null) {
			if (venue.id != null) vars.put("vid", venue.id);
			else vars.put("venue", venue.name);
		}
		if (shout!=null) vars.put("shout", shout);
		// privacy settings
		if (sendToTwitter!=null) vars.put("twitter", sendToTwitter? "1" : "0");
		if (sendToFacebook!=null) vars.put("facebook", sendToFacebook? "1" : "0");
		if (hide) vars.put("private", "1");
		String json = http.getPage(API_URL+"checkin", vars, true);
		try {
			return new CheckIn(new JSONObject(json), self);
		} catch (JSONException e) {
			throw new FoursquareException(e);
		}
	}
	
	/**
	 * Set to override the default privacy settings for check-ins.
	 * @param sendToTwitter If true, tweet on check-in. If false, don't. If unset - use your foursquare default settings.
	 * @param sendToFacebook If true, update facebook on check-in. If false, don't. If unset - use your foursquare default settings.
	 * @param hide If true, make a private check-in which your friends can't see (e.g. for dates -- but why are you fiddling with foursquare then?).
	 */
	public void setCheckInPrivacy(Boolean sendToTwitter, Boolean sendToFacebook, boolean hide) 
	{
		this.sendToTwitter = sendToTwitter;
		this.sendToFacebook = sendToFacebook;
		this.hide = hide;
	}
	
	/**
	 * This will call foursquare once then cache the result.
	 * @return
	 */
	public User4Sq getSelf() {
		if (self==null) {
			self = getUser(null, null, true);
		}
		return self;
	}
	
	
	public User4Sq getUser(String userId, String twitterName, boolean getExtraInfo) 
	{
		Map vars = getStandardishParameters();
		vars.put("twitter", twitterName);
		vars.put("uid", userId);
		if (getExtraInfo) {
			vars.put("badges", "1");
			vars.put("mayor", "1");
		}
		String json = http.getPage(API_URL+"user.json", vars, true);
		try {
			JSONObject uo = new JSONObject(json).getJSONObject("user");
			return new User4Sq(uo);
		} catch (JSONException e) {
			throw new FoursquareException(e);
		}
	}
	
	private Map<String, Object> getStandardishParameters() 
	{
		return addStandardishParameters(new HashMap<String, Object>());
	}
	
	/**
	 * Add in since_id, page and count, if set. This is called by methods that return
	 * lists of statuses or messages.
	 *
	 * @param vars
	 * @return vars
	 */
	private Map<String, Object> addStandardishParameters(
			Map<String, Object> vars) 
	{
		if (sinceId != null)
			vars.put("sinceid", sinceId);
		if (maxResults > 0) {
			vars.put("l", maxResults);
		}
		if (location!=null) {
			vars.put("geolat", location.latitude);
			vars.put("geolong", location.longitude);
		}
		return vars;
	}
	
	public void setMaxResults(int maxResults) {
		assert maxResults <= 250;
		this.maxResults = maxResults;
	}

	private Long sinceId;

	
	/**
	 * Provides support for fetching many pages
	 */
	private int maxResults;

	private User4Sq self;

	private double hAccuracy;

	private double vAccuracy;

	private Locn location;
	
	/**
	 *
	 * @param url
	 * @param var
	 * @param authenticate
	 * @return
	 */
	private List<CheckIn> getCheckIns(String url, Map<String, Object> var) 
	{
		String json = http.getPage(url, (Map)var, true);
		List<CheckIn> msgs = CheckIn.parseJsonList(json, self);
		return msgs;
	}
	
	/**
	 * 
	 * @param latitude
	 * @param longitude
	 * @param altitude In metres. NOT USED YET
	 */
	public void setLocation(Locn locn) 
	{
		this.location = locn;
	}
	
	/**
	 * 
	 * @param horizontal In metres
	 * @param vertical In metres
	 */
	public void setLocationAccuracy(double horizontal, double vertical) {
		this.hAccuracy = horizontal;
		this.vAccuracy = vertical;
	}
	
	/**
	 * Java wrapper for the foursquare API.
	 *
	 * @param name
	 *            the authenticating user's name, if known. Can be null.
	 * @param client
	 */
	public J4Square(IHttpClient client) {
		http = client;
	}
	
	public J4Square(String phoneOrEmail, String password) {
		this(new URLConnectionHttpClient(phoneOrEmail, password));
	}
	
	public static void main(String[] args) {
		System.out.println("Java interface for foursquare");
		System.out.println("--------------------------");
		System.out.println("Version "+version);
		System.out.println("Released under GPL by Winterwell Associates Ltd.");
		System.out
		.println("See source code or JavaDoc for details on how to use.");
	}

	public Tip addTip(Venue venue, String msg) {
		assert venue != null && venue.id != null;
		assert msg != null;
		Map<String,String> vars = asMap("vid", venue.id, "text", msg);
		String json = http.post(API_URL+"addtip.json", vars, true);
		try {
			JSONObject jo = new JSONObject(json);
			JSONObject tip = jo.getJSONObject("tip");
			return new Tip(tip);
		} catch (JSONException e) {
			throw new FoursquareException.Parsing(json, e);
		}
	}
	
	public List<Venue> venues(String query) {
		Map vars = getStandardishParameters();
		if (query!=null) vars.put("q", query);
		String json = http.getPage(API_URL + "venues.json", vars, http.canAuthenticate());
		return Venue.parse(json);
	}
	
	
	public static final class Locn {
		public final double longitude;
		public final double latitude;
		public final double altitude = 0;
		public Locn(double latitude, double longitude) {
			this.longitude = longitude;
			this.latitude = latitude;
		}
	}
	
	/**
	 * Create a map from a list of key, value pairs. An easy way to make small
	 * maps, basically the equivalent of {@link Arrays#asList(Object...)}.
	 */
	@SuppressWarnings("unchecked")
	static <K, V> Map<K, V> asMap(Object... keyValuePairs) {
		assert keyValuePairs.length % 2 == 0;
		Map m = new HashMap(keyValuePairs.length / 2);
		for (int i = 0; i < keyValuePairs.length; i += 2) {
			m.put(keyValuePairs[i], keyValuePairs[i + 1]);
		}
		return m;
	}
}
